<?php


class AWE_Thumbnail {

	public $id;

	public $title;

	public $screen;

	public $context;

	public $priority;

	public $fields;


	/**
	 * @param 	array 	post type to show meta box
	 *
	 * @return 	void
	 *
	 * @access 	public
	 * @since 	1.0
	 *
	 */
	public function __construct( $args = array() ) {


		$default_args = array(
			'id'	=> 'awe-thumbnail',
			'title'	=> __( 'AWE Thumbnail Setting', 'hillter' ),
			'screen'	=> array( 'post' ),
			'context'	=> 'normal',
			'priority'	=> 'low',
			'fields'	=> array(),
		);
		$args = wp_parse_args( $args, $default_args );

		$this->id = esc_attr( $args['id'] );
		$this->title = esc_html( $args['title'] );
		$this->screen = ( array ) $args['screen'];
		$this->context = in_array( $args['context'], array( 'normal', 'advanced', 'side' ) ) ? $args['context'] : $default_args['context'];
		$this->priority = in_array( $args['priority'], array( 'high', 'core', 'default', 'low' ) ) ? $args['priority'] : $default_args['priority'];
		$this->fields = ( array ) $args['fields'];

		// Add meta box
		add_action('add_meta_boxes', array($this, 'add_meta_box'));

		// Save meta value
		add_action('save_post', array($this, 'save'));

		// Load css & js
		add_action('admin_enqueue_scripts', array($this, 'load_assets'));
	}


	/**
	 * Load css & js
	 *
	 * @return 	void\
	 *
	 * @access 	public
	 * @since 	1.0
	 *
	 */
	public function load_assets() {
		wp_enqueue_style( 'awe-thumbnail-css', AWE_THUMBNAIL_CSS_PATH . 'style.css' );
		wp_enqueue_script( 'awe-thumbnail-js', AWE_THUMBNAIL_JS_PATH . 'script.js', array( 'jquery' ), false, true );
	}


	/**
	 * Add meta box
	 *
	 * @param 	string
	 *
	 * @return 	void
	 *
	 * @access 	public
	 * @since 	1.0
	 *
	 */
	public function add_meta_box( $post_type ) {
		if ( in_array( $post_type, $this->screen ) ) {
			add_meta_box( $this->id, $this->title, array( $this, 'render' ), $post_type, $this->context, $this->priority );
		}
	}


	/**
	 * Save meta value
	 *
	 * @param 	int 	Post ID
	 *
	 * @return 	void|int 	Post ID
	 *
	 * @access public
	 * @since 	1.0
	 *
	 */
	public function save( $post_id ) {
		// Check if our nonce is set.
		if ( ! isset( $_POST['awe_' . $this->id . '_nonce_field'] ) )
			return $post_id;

		$nonce = $_POST['awe_' . $this->id . '_nonce_field'];

		// Verify that the nonce is valid.
		if ( ! wp_verify_nonce( $nonce, 'awe_' . $this->id . '_nonce' ) )
			return $post_id;

		// If this is an autosave
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE )
			return $post_id;

		// Check the user's permissions.
		if ( ! current_user_can( 'edit_post', $post_id ) )
			return $post_id;

		// All OK

		// Thumbnail type
		$key = $this->id . '-type';
		$value = isset( $_POST[$key] ) && isset( $this->fields[$_POST[$key]] ) ? $_POST[$key] : 'feature';
		update_post_meta( $post_id, $key, $value );

		// Other field
		foreach ( $this->fields as $k => $v ) {
			// meta key
			$key = $this->id . '-' . $k;

			switch ( $v['type'] ) {


				// Single image
				case 'image':

					$value = isset( $_POST[$key] ) ? esc_url( $_POST[$key] ) : '';
					update_post_meta( $post_id, $key, $value );

				break;


				// gallery
				case 'gallery':

					$value = isset( $_POST[$key] ) ? wp_parse_args( $_POST[$key], array( 'shortcode' => '', 'images' => array() ) ) : '';
					if ( ! empty( $value['shortcode'] ) && ! empty( $value['images'] ) ) {
						$value['shortcode'] = esc_html( $value['shortcode'] );
						$value['images'] = array_map( 'esc_url', $value['images'] );
					}
					update_post_meta( $post_id, $key, $value );

				break;


				// Video
				case 'video':

					$value = isset( $_POST[$key] ) ? wp_parse_args( $_POST[$key], array( 'url' => '', 'embed' => '' ) ) : '';
					if ( ! empty( $value['url'] ) && ! empty( $value['embed'] ) ) {
						$value = array_map( 'esc_url', $value );
					}
					update_post_meta( $post_id, $key, $value );

				break;


				// Audio
				case 'audio':

					$value = isset( $_POST[$key] ) ? wp_parse_args( $_POST[$key], array( 'url' => '', 'autoplay' => 'false' ) ) : '';
					if ( ! empty( $value['url'] ) ) {
						$value['url'] = esc_url( $value['url'] );
						$value['autoplay'] = in_array( $value['autoplay'], array( 'true', 'false' ) ) ? $value['autoplay'] : 'false';
					}
					update_post_meta( $post_id, $key, $value );

				break;

			}

		}

	}


	/**
	 * Display meta box
	 *
	 * @param 	WP_Post 	Post object
	 *
	 * @return 	void
	 *
	 * @access public
	 * @since 	1.0
	 *
	 */
	public function render( $post ) {
		wp_nonce_field( 'awe_' . $this->id . '_nonce', 'awe_' . $this->id . '_nonce_field' );

		// Thumbnail type
		$type = get_post_meta( $post->ID, $this->id . '-type', true );
		$type = $type && isset( $this->fields[$type] ) ? $type : 'feature';
		?>

		<div id="awe-thumbnail-<?php echo esc_attr( $this->id ); ?>-container" class="awe-thumbnail-meta-box-container">

			<div class="awe-thumbnail-type">
				<select name="<?php echo esc_attr( $this->id ); ?>-type" id="<?php echo esc_attr( $this->id ); ?>-type">
					<option value="feature" <?php selected( $type, 'feature' ); ?>><?php _e( 'As Feature Image', 'hillter' ); ?></option>
					<?php foreach ( $this->fields as $k => $v ) : ?>
						<option value="<?php echo esc_attr( $k ); ?>" <?php selected( $type, $k ); ?>><?php echo esc_html( $v['title'] ); ?></option>
					<?php endforeach; ?>
				</select>
			</div>

			<?php

			foreach ( $this->fields as $k => $v ) {

				echo '<div class="awe-thumbnail-' . esc_attr( $k ) . '"' . ( $type != $k ? ' style="display:none;"' : '' ) . '>';

				// Meta key
				$key = $this->id . '-' . $k;

				// Meta value
				$value = get_post_meta( $post->ID, $key, true );

				switch ( $v['type'] ) {

					// Single image
					case 'image':
						$value = esc_url( $value );
						?>

							<div class="image-view">
								<?php if ( $value ) : ?>
									<img src="<?php echo esc_url( $value ); ?>">
								<?php else : ?>
									<img src="<?php echo esc_url( AWE_THUMBNAIL_IMG_PATH . 'no-image.jpg' ); ?>">
								<?php endif; ?>
							</div>

							<div class="control">
								<button type="button" class="button button-primary single-image-upload-btn"><?php _e( 'Choose Image', 'hillter' ); ?></button>
								<button type="button" class="button single-image-remove-btn" data-no-image="<?php echo esc_url( AWE_THUMBNAIL_IMG_PATH . 'no-image.jpg' ); ?>"><?php _e( 'Remove', 'hillter' ); ?></button>
							</div>

							<input type="hidden" name="<?php echo esc_attr( $key ); ?>" value="<?php echo esc_url( $value ); ?>">

						<?php

					break;


					// gallery
					case 'gallery':
						$value = wp_parse_args( $value, array( 'shortcode' => '', 'images' => array() ) );

						// filter value
						$value['shortcode'] = esc_html( $value['shortcode'] );
						$value['images'] = array_map( 'esc_url', $value['images'] );
						$list_id = array();
						?>

							<div class="images-view">
								<?php
								if ( ! empty( $value['images'] ) ) {
									foreach ( $value['images'] as $k => $v ) {
										// store attachment id to edit
										$list_id[] = absint( $k );

										echo '<img src="' . esc_url( $v ) . '">';
										echo '<input type="hidden" name="' . esc_attr( $key ) . '[images][' . absint( $k ) . ']" value="' . esc_url( $v ) . '">';
									}
								} else {
									echo '<img src="' . esc_url( AWE_THUMBNAIL_IMG_PATH . 'no-image.jpg' ) . '">';
								}

								$list_id = implode( ',', $list_id );
								?>

							</div>

							<div class="control">
								<button type="button" class="button button-primary gallery-upload-btn" data-key="<?php echo esc_attr( $key ); ?>"><?php _e( 'Choose Image', 'hillter' ); ?></button>
								<button type="button" class="button gallery-edit-btn" data-list-id="<?php echo esc_attr( $list_id ); ?>"><?php _e( 'Edit', 'hillter' ); ?></button>
								<button type="button" class="button gallery-remove-btn" data-no-image="<?php echo esc_url( AWE_THUMBNAIL_IMG_PATH . 'no-image.jpg' ); ?>"><?php _e( 'Remove', 'hillter' ); ?></button>
							</div>

							<input type="hidden" name="<?php echo esc_attr( $key ); ?>[shortcode]" value="<?php echo esc_html( $value['shortcode'] ); ?>">

						<?php

					break;


					// Video
					case 'video':
						$value = wp_parse_args( $value, array( 'url' => '', 'embed' => '' ) );
						$value = array_map( 'esc_url', $value );
						?>

							<div class="video-view">
								<?php if ( $value['url'] ) : ?>
									<div class="embed-responsive embed-responsive-16by9">
										<iframe class="embed-responsive-item" src="<?php echo esc_url( $value['embed'] ); ?>"></iframe>
									</div>
								<?php else : ?>
									<img src="<?php echo esc_url( AWE_THUMBNAIL_IMG_PATH . 'no-video.gif' ); ?>">
								<?php endif; ?>
							</div>

							<div class="control">
								<input type="text" name="<?php echo esc_attr( $key ); ?>[url]" value="<?php echo esc_url( $value['url'] ); ?>" size="40" placeholder="<?php _e( 'Enter Youtube or Vimeo video url', 'hillter' ); ?>">
								<button type="button" class="button button-primary video-add-btn"><?php _e( 'Choose Video', 'hillter' ); ?></button>
								<button type="button" class="button video-remove-btn" data-no-video="<?php echo esc_url( AWE_THUMBNAIL_IMG_PATH . 'no-video.gif' ); ?>"><?php _e( 'Remove', 'hillter' ); ?></button>
								<p>
									<?php _e( 'Vimeo url like:', 'hillter' ); ?> <strong>http://vimeo.com/27260633</strong><br />
									<?php _e( 'Youtube url like:', 'hillter' ); ?> <strong>https://www.youtube.com/watch?v=KV2ssT8lzj8</strong>
								</p>
							</div>

							<input type="hidden" name="<?php echo esc_attr( $key ); ?>[embed]" value="<?php echo esc_url( $value['embed'] ); ?>">

						<?php

					break;


					// Audio
					case 'audio':
						$value = wp_parse_args( $value, array( 'url' => '', 'autoplay' => 'false' ) );
						$value['url'] = esc_url( $value['url'] );
						$value['autoplay'] = in_array( $value['autoplay'], array( 'true', 'false' ) ) ? $value['autoplay'] : 'false';
						?>

							<div class="audio-view">
								<?php if ( $value['url'] ) : ?>
									<?php
									if (preg_match("/\/sets\//i", $value['url']) || preg_match("/\/sets\//i", $value['url'])) {
			                            $maxheight = "450px";
									} else {
			                            $maxheight = "166px";
			                        }
			                        ?>
									<iframe width="100%" height="<?php echo esc_attr( $maxheight ); ?>" scrolling="no" frameborder="no" src="https://w.soundcloud.com/player/?url=<?php echo esc_url( $value['url'] ); ?>&amp;auto_play=<?php echo esc_attr( $value['autoplay'] ); ?>&amp;hide_related=false&amp;show_comments=true&amp;show_user=true&amp;show_reposts=false&amp;visual=true"></iframe>
								<?php else : ?>
									<img src="<?php echo esc_url( AWE_THUMBNAIL_IMG_PATH . 'no-audio.jpg' ); ?>">
								<?php endif; ?>
							</div>

							<div class="control">
								<input type="text" name="<?php echo esc_attr( $key ); ?>[url]" value="<?php echo esc_url( $value['url'] ); ?>" size="40" placeholder="<?php _e( 'Enter Soundcloud audio url', 'hillter' ); ?>">
								<button type="button" class="button button-primary audio-add-btn"><?php _e( 'Choose Audio', 'hillter' ); ?></button>
								<button type="button" class="button audio-remove-btn" data-no-audio="<?php echo esc_url( AWE_THUMBNAIL_IMG_PATH . 'no-audio.jpg' ); ?>"><?php _e( 'Remove', 'hillter' ); ?></button>
								<br>
								<label><input type="checkbox" name="<?php echo esc_attr( $key ); ?>[autoplay]" value="true" <?php checked( $value['autoplay'], "true" ); ?>> <?php _e( 'Auto play', 'hillter' ); ?></label>
							</div>

						<?php

					break;

				} // end switch

				echo '</div>';

			}

			?>

			<script type="text/javascript">
				(function($) {
					"use strict";

					$(document).ready(function() {
						var $metaBox = $("#awe-thumbnail-<?php echo esc_attr( $this->id ); ?>-container");
						var $selectType = $("#<?php echo esc_attr( $this->id ); ?>-type");

						$selectType.on('change', function() {
							var value = $(this).val();

							$metaBox.children(":not(.awe-thumbnail-type)").hide();
							$metaBox.children(".awe-thumbnail-" + value).fadeIn();
						});

					});
				})(jQuery);
			</script>

		</div>

		<?php
		/***** End Feature Thumbnail *****/
	}

}
