<?php

namespace TablePress\PhpOffice\PhpSpreadsheet\Calculation\MathTrig;

use TablePress\PhpOffice\PhpSpreadsheet\Calculation\ArrayEnabled;
use TablePress\PhpOffice\PhpSpreadsheet\Calculation\Exception;

class Trunc
{
	use ArrayEnabled;

	/**
	 * TRUNC.
	 *
	 * Truncates value to the number of fractional digits by number_digits.
	 * This will probably not be the precise result in the unlikely
	 * event that the number of digits to the left of the decimal
	 * plus the number of digits to the right exceeds PHP_FLOAT_DIG
	 * (or possibly that value minus 1).
	 * Excel is unlikely to do any better.
	 *
	 * @param array|float $value Or can be an array of values
	 * @param array|int $digits Or can be an array of values
	 *
	 * @return array|float|string Truncated value, or a string containing an error
	 *         If an array of numbers is passed as an argument, then the returned result will also be an array
	 *            with the same dimensions
	 */
	public static function evaluate($value = 0, $digits = 0)
	{
		if (is_array($value) || is_array($digits)) {
			return self::evaluateArrayArguments([self::class, __FUNCTION__], $value, $digits);
		}

		try {
			$value = Helpers::validateNumericNullBool($value);
			$digits = Helpers::validateNumericNullSubstitution($digits, null);
		} catch (Exception $e) {
			return $e->getMessage();
		}

		if ($value == 0) {
			return $value;
		}

		if ($value >= 0) {
			$minusSign = '';
		} else {
			$minusSign = '-';
			$value = -$value;
		}
		$digits = (int) floor($digits);
		if ($digits < 0) {
			$result = (float) (substr(sprintf('%.0F', $value), 0, $digits) . str_repeat('0', -$digits));

			return ($minusSign === '') ? $result : -$result;
		}
		$decimals = (floor($value) == (int) $value) ? (PHP_FLOAT_DIG - strlen((string) (int) $value)) : $digits;
		$resultString = ($decimals < 0) ? sprintf('%F', $value) : sprintf('%.' . $decimals . 'F', $value);
		$regExp = '/([.]\\d{' . $digits . '})\\d+$/';
		$result = $minusSign . (preg_replace($regExp, '$1', $resultString) ?? $resultString);

		return (float) $result;
	}
}
